#!/bin/zsh

###############################################################################
# Script Name   : add-image-ids.command
# Description   : Script to embed 32 character image IDs in metadata of
#                 images in a user-selected folder
# Author        : Bill Anderson
# Usage License : Root Beer Tag Client Software EULA
#                 https://www.rootbeertag.com/legal/client-eula
# Copyright     : (c) 2024-2025 RBT Software LLC
###############################################################################

#set -x

# Set the terminal window size
printf '\e[8;35;140t'
osascript -e 'tell application "Terminal" to set position of front window to {22, 44}'
osascript -e 'tell application "Terminal" to set the font size of front window to 14'

IFS=$(echo -en "\n\b")

BASE_DIR=$HOME/.rootbeertag
if [ ! -d $BASE_DIR ]; then
    mkdir -p $BASE_DIR
fi

THIS_FULL_PATH=${(%):-%N}
THIS_DIR=$(dirname "$THIS_FULL_PATH")

FIRST_RUN_TEXT=`cat $THIS_DIR/first-run.txt`
CONFIG_FILE=$BASE_DIR/config
if [ -f $CONFIG_FILE ]; then
    LAST_IMAGE_BASE_DIR=`cat ${CONFIG_FILE} | grep last_image_base_dir | cut -d ":" -f2`
    if [ ! -d ${LAST_IMAGE_BASE_DIR} ]; then
        LAST_IMAGE_BASE_DIR=/users/$USER/Pictures
    fi
else
    LAST_IMAGE_BASE_DIR=/users/$USER/Pictures
    WELCOME="$(osascript -e "display dialog \"$FIRST_RUN_TEXT\" with title \"Welcome to Root Beer Tag!\"")"
    if [ "$WELCOME" != "button returned:OK" ]; then
        echo; echo "Execution canceled; this window may be closed.";echo
        exit 1
    fi
fi

CURRENT_IMAGE_BASE_DIR="$(osascript -l JavaScript -e 'a=Application.currentApplication();a.includeStandardAdditions=true;a.chooseFolder({withPrompt:"Please select a folder to search for images:",defaultLocation:"'"$LAST_IMAGE_BASE_DIR"'"}).toString()')"

if [ ! $CURRENT_IMAGE_BASE_DIR ]; then
    echo; echo "Execution canceled; this window may be closed."; echo
    exit 1
fi

COUNT_IMAGES_TO_PROCESS=`find $CURRENT_IMAGE_BASE_DIR -type f \( -iname "*.jpg" -o -iname "*.jpeg" -o -iname "*.png" -o -iname "*.gif" \)| grep -v "Photos Library" | wc -l | sed -e 's/^[ \t]*//'`

if [ $COUNT_IMAGES_TO_PROCESS -eq 0 ]; then
    echo
    echo "There are no images to process; this window may be closed."
    echo
    exit 0
fi

echo;echo "All ${COUNT_IMAGES_TO_PROCESS} images in ${CURRENT_IMAGE_BASE_DIR} will be processed."

# This little bit of insanity does a little data sanitizing prior of the directories and filenames.

SPEC_CHAR_DIR_COUNT=`find $CURRENT_IMAGE_BASE_DIR -type d | grep "[^]^A-Za-z0-9\.\(\)\/_ ,-\&+\!{}~@$]" | wc -l | sed -e 's/^[ \t]*//'`

echo;echo "Removing special characters from $SPEC_CHAR_DIR_COUNT directory names."

while [ $SPEC_CHAR_DIR_COUNT -gt 0 ]
do
    for SPEC_CHAR_CURR_DIR in `find $CURRENT_IMAGE_BASE_DIR -type d | grep "[^]^A-Za-z0-9\.\(\)\/_ ,-\&+\!{}~@$]"`
    do
        SPEC_CHAR_NEW_DIR=`echo $SPEC_CHAR_CURR_DIR | sed -e 's/[^]^A-Za-z0-9\.\(\)\/_ ,-\&+\!{}~@$]//g'`
        mv $SPEC_CHAR_CURR_DIR $SPEC_CHAR_NEW_DIR
    done
    SPEC_CHAR_DIR_COUNT=`find $CURRENT_IMAGE_BASE_DIR -type d | grep "[^]^A-Za-z0-9\.\(\)\/_ ,-\&+\!{}~@$]" | wc -l | sed -e 's/^[ \t]*//'```
done

SPEC_CHAR_FILE_COUNT=`find $CURRENT_IMAGE_BASE_DIR -type f | grep "[^]^A-Za-z0-9\.\(\)\/_ ,-\&+\!{}~@$]" | wc -l | sed -e 's/^[ \t]*//'`

echo "Removing special characters from $SPEC_CHAR_FILE_COUNT file names."

while [ $SPEC_CHAR_FILE_COUNT -gt 0 ]
do
    for SPEC_CHAR_CURR_FILE in `find $CURRENT_IMAGE_BASE_DIR -type f | grep "[^]^A-Za-z0-9\.\(\)\/_ ,-\&+\!{}~@$]"`
    do
        SPEC_CHAR_NEW_FILE=`echo $SPEC_CHAR_CURR_FILE | sed -e 's/[^]^A-Za-z0-9\.\(\)\/_ ,-\&+\!{}~@$]//g'`
        mv $SPEC_CHAR_CURR_FILE $SPEC_CHAR_NEW_FILE
    done
    SPEC_CHAR_FILE_COUNT=`find $CURRENT_IMAGE_BASE_DIR -type f | grep "[^]^A-Za-z0-9\.\(\)\/_ ,-\&+\!{}~@$]" | wc -l | sed -e 's/^[ \t]*//'`
done

IMAGES_TO_PROCESS=`find $CURRENT_IMAGE_BASE_DIR -type f \( -iname "*.jpg" -o -iname "*.jpeg" -o -iname "*.png" -o -iname "*.gif" \)| grep -v "Photos Library"`

COUNT_IMAGES_TO_PROCESS=`find $CURRENT_IMAGE_BASE_DIR -type f \( -iname "*.jpg" -o -iname "*.jpeg" -o -iname "*.png" -o -iname "*.gif" \)| grep -v "Photos Library" | wc -l | sed -e 's/^[ \t]*//'`

echo

echo "Verifying and/or generating and embedding image IDs in ${COUNT_IMAGES_TO_PROCESS} image(s)."
echo "As your images are processed, only images that need metadata modification will be listed."
echo "This process can take a little while to roll through large image collections."

echo

exiftool -if '$otherimagestart and $make eq "samsung"' -exif:all= -tagsfromfile @ -all:all -unsafe -execute -if '$imageuniqueid !~ /^[A-F0-9]{32}$$/' '-imageuniqueid<NewGUID' -common_args -overwrite_original_in_place -m -q -q -progress -recurse -n -i 'Photos Library' -ext jpg -ext jpeg -ext gif -ext png $CURRENT_IMAGE_BASE_DIR

echo

echo "Processing complete; ${COUNT_IMAGES_TO_PROCESS} images(s) processed"

# Last but not least, write out the config file for next time.

echo "last_image_base_dir:${CURRENT_IMAGE_BASE_DIR}" > $CONFIG_FILE

echo

echo "This window may be closed.  Thank you!"

exit 0

# zip rbt-macos-client.zip ExifTool-13.30.pkg add-image-ids.command remove-image-ids.command first-run.txt remove-readme.txt license.txt