#!/bin/zsh

##################################################################################
# Script Name   : remove-image-ids.command
# Description   : Script to remove ImageUniqueID metadata from selected image(s).
# Author        : Bill Anderson
# Usage License : Root Beer Tag Client Software EULA
#                 https://www.rootbeertag.com/legal/client-eula
# Copyright     : (c) 2025-2025 RBT Software LLC
##################################################################################

#set -x

# Set the terminal window size
printf '\e[8;35;140t'
osascript -e 'tell application "Terminal" to set position of front window to {22, 44}'
osascript -e 'tell application "Terminal" to set the font size of front window to 14'

IFS=$(echo -en "\n\b")

#BASE_DIR=$HOME/.rootbeertag
#if [ ! -d $BASE_DIR ]; then
#    mkdir -p $BASE_DIR
#fi

THIS_FULL_PATH=${(%):-%N}
THIS_DIR=$(dirname "$THIS_FULL_PATH")

README_TEXT=`cat $THIS_DIR/remove-readme.txt`
IMAGE_LIST_FILE=$THIS_DIR/image-list.txt
#CONFIG_FILE=$BASE_DIR/config
IMAGE_BASE_DIR=/users/$USER/Pictures

WELCOME="$(osascript -e "display dialog \"$README_TEXT\" with title \"!!! CAUTION !!!\"")"
if [ "$WELCOME" != "button returned:OK" ]; then
    echo; echo "Execution canceled; this window may be closed.";echo
    exit 1
fi

IMAGES="$(osascript -l JavaScript -e 'a=Application.currentApplication();a.includeStandardAdditions=true;a.chooseFile({withPrompt:"Select Image(s)",multipleSelectionsAllowed:"true",defaultLocation:"'"$IMAGE_BASE_DIR"'"}).join("\n")')"

if [ ! $IMAGES ]; then
    echo; echo "Execution canceled; this window may be closed."; echo
    exit 1
fi

IMAGE_COUNT=`echo $IMAGES | wc -l | sed -e 's/^ *//'`

if [ $IMAGE_COUNT -eq 0 ]; then
    echo
    echo "There are no images to process; this window may be closed."
    echo
    exit 0
fi

# I know this next part is jankey as hell, but it's the only I could get it work nicely with the OSA output.
echo $IMAGES > $IMAGE_LIST_FILE

# Show the user what they've selected.

echo ""
echo "You have selected the following $IMAGE_COUNT image(s)."
echo ""
echo $IMAGES

#  Ask the user what they want to do with what they've selected.

echo ""
echo "What do you wish to do with these images' unique IDs (if they exist)?"
echo ""
echo "1. Remove the unique image IDs from their metadata."
echo "2. Replace the unique image IDs in their metadata with new random IDs."
echo "3. Leave the images and their metadata unchanged."

unset FINAL_RESPONSE

while [ -z $FINAL_RESPONSE ]
do
    echo ""
    echo -n "Please enter your preferred option (1, 2, or 3): "
    read RESPONSE

    if [[ "$RESPONSE" != "1" && "$RESPONSE" != "2" && "$RESPONSE" != "3" ]]; then
      echo ""
      echo "Invalid option. Valid options are only the numbers 1, 2, and 3."
    else
        FINAL_RESPONSE=$RESPONSE
    fi
done

case $FINAL_RESPONSE in 
    1)
        echo ""
        echo "Removing image IDs from $IMAGE_COUNT image(s) if they exist."
        echo ""
        echo "As images are processed, only images that currently have a unique image ID will be listed as those IDs are removed."
        echo ""
        exiftool -overwrite_original_in_place -n -if '$imageuniqueid' -imageuniqueid= -progress -m -q -q `cat ${IMAGE_LIST_FILE}`
        echo ""
        echo "All $IMAGE_COUNT images have been processed."
        echo ""
        echo "This window may be closed.  Thank you!"
    ;;
    2)
        echo ""
        echo "Replacing current image IDs in $IMAGE_COUNT image(s) if they exist"
        echo ""
        echo "As images are processed, only images that currently have a unique image ID will be listed as those IDs are removed."
        echo ""
        exiftool -overwrite_original_in_place -n -if '$imageuniqueid' '-imageuniqueid<newGUID' -progress -m -q -q `cat ${IMAGE_LIST_FILE}`
        echo ""
        echo "All $IMAGE_COUNT images have been processed."
        echo ""
        echo "This window may be closed.  Thank you!"
    ;;
    3)
        echo ""
        echo "Exiting per user request.  No images have been modified."
        echo ""
        echo "This window may be closed.  Thank you!"
    ;;
esac

if [ -f $IMAGE_LIST_FILE ]; then
    rm $IMAGE_LIST_FILE
fi

exit 0

# zip rbt-macos-client.zip ExifTool-13.30.pkg add-image-ids.command remove-image-ids.command first-run.txt remove-readme.txt license.txt